#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime

from django.db import models
from django.core import exceptions
from django.contrib.auth import get_user_model
from django.conf import settings
from django.utils import timezone
from django.core.validators import MaxValueValidator, MinValueValidator
from django.template import TemplateDoesNotExist, engines
from django.utils.translation import gettext_lazy as _
from solo.models import SingletonModel
from django.template.loader import get_template
from oscar.core.loading import get_model

User = get_user_model()
Source = get_model('payment', 'Source')
UserAddress = get_model('address', 'UserAddress')
Order = get_model('order', 'Order')
ShippingAddress = get_model('order', 'ShippingAddress')

ATM = 'ATM'

TEXT = "text"
INTEGER = "integer"
BOOLEAN = "boolean"
FLOAT = "float"
RICHTEXT = "richtext"
DATE = "date"
DATETIME = "datetime"
OPTION = "option"
MULTI_OPTION = "multi_option"
ENTITY = "entity"
FILE = "file"
IMAGE = "image"
TYPE_CHOICES = (
    (TEXT, _("Text")),
    (INTEGER, _("Integer")),
    (BOOLEAN, _("True / False")),
    (FLOAT, _("Float")),
    (RICHTEXT, _("Rich Text")),
    (DATE, _("Date")),
    (DATETIME, _("Datetime")),
    (OPTION, _("Option")),
    #(MULTI_OPTION, _("Multi Option")),
    #(ENTITY, _("Entity")),
    (FILE, _("File")),
    (IMAGE, _("Image")),
)

class OptionType(models.Model):
    option = models.ForeignKey(
        'catalogue.Option',
        on_delete=models.CASCADE,
        verbose_name=_("Option")
    )
    
    type = models.CharField(
        choices=TYPE_CHOICES, default=TYPE_CHOICES[0][0],
        max_length=20, verbose_name=_("Type"))
    optionGroup = models.ForeignKey(
        'catalogue.AttributeOptionGroup',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
        related_name='productAttributes',
        verbose_name=_("Option Group"),
        help_text=_(
            'Select an option group if using type "Option" or "Multi Option"'))
            
class OrdererAddress(models.Model):
    user = models.ForeignKey(
        User,
        related_name='orderer',
        verbose_name=_("Orderer"),
    )
    
    address = models.ForeignKey(
        'address.UserAddress',
        on_delete=models.CASCADE,
        related_name='ordererAddress',
        verbose_name=_("Orderer Address"),
    )
    
@property
def ordererAddress(self):
    if OrdererAddress.objects.filter(user=self).exists():
        return OrdererAddress.objects.get(user=self).address
    else:
        return None
        
User.ordererAddress = ordererAddress
    
class OrderOrdererAddress(models.Model):
    order = models.ForeignKey(
        Order,
        related_name='order',
        verbose_name=_("Order"),
    )
    
    address = models.ForeignKey(
        ShippingAddress,
        on_delete=models.CASCADE,
        related_name='ordererAddress',
        verbose_name=_("Orderer Address"),
    )
    
@property
def ordererAddress2(self):
    if OrderOrdererAddress.objects.filter(order=self).exists():
        return OrderOrdererAddress.objects.get(order=self).address
    else:
        return None

Order.ordererAddress = ordererAddress2

# https://code.nuwainfo.com/trac/Mercurius/ticket/3296#ticket
@property
def useATM(self):
    
    if Source.objects.filter(order=self, source_type__name=ATM).exists():
        source = Source.objects.get(order=self, source_type__name=ATM)
        return source
    else:
        return None
        
@property
def useATMPaymentSuccess(self):
    source = self.useATM
    if source:
        if not source.reference == '-':
            return True
        else:
            return False
    else:
        return False
        
@property
def useATMPaymentExpired(self):
    expiredHours = settings.IUNO_SHOP_ATM_EXPIRED_HOURS
    expiredHours = datetime.timedelta(hours=expiredHours)
    now = timezone.localtime(timezone.now())
    
    if now - self.date_placed > expiredHours:
        return True
    return False
        
def setATMCode(self, code):
    source = self.useATM
    if source:
        source.reference = code
        source.save()

Order.useATM = useATM
Order.useATMPaymentSuccess = useATMPaymentSuccess
Order.useATMPaymentExpired = useATMPaymentExpired
Order.setATMCode = setATMCode    
    
@property
def isOrdererAddress(self):
    orderAddress, created = OrdererAddress.objects.get_or_create(
        user=self.user,
        defaults={
            'address': self,
        }
    )
    if orderAddress.address == self:
        return True
    else:
        return False
        
UserAddress.isOrdererAddress = isOrdererAddress

class Bonus(models.Model):
    user = models.ForeignKey(
        User,
        verbose_name=_("User"),
    )
    
    quantity = models.PositiveIntegerField(
        verbose_name=_("Quantity"), default=0)
        
@property
def bonus(self):
    b, created = Bonus.objects.get_or_create(user=self)
    return b.quantity
    
User.bonus = bonus
    
class BonusRecord(models.Model):
    user = models.ForeignKey(
        User,
        verbose_name=_("User"),
    )
    order = models.ForeignKey(
        Order,
        verbose_name=_("Order"),
        null=True,
    )
    
    
    use = models.PositiveIntegerField(
        verbose_name=_("Use"), default=0)
    receive = models.PositiveIntegerField(
        verbose_name=_("Receive"), default=0)
    adjust = models.IntegerField(
        verbose_name=_("Adjust"), default=0)
        
    createdTime = models.DateTimeField(_("Created Time"), auto_now_add=True)
    
@property
def hasReceiveBonus(self):
    return BonusRecord.objects.filter(order=self).exists()
    
def calcReceiveBouns(total, shipping):
    config = BonusSettings.objects.get()
    
    if shipping:
        consumption = (
            int(total) - config.consumption - 
            int(shipping))
    else:
        consumption = (
            int(total) - config.consumption)
            
    receive = 0
    if consumption >= 0:
        # config.percent / 100 = X %
        receive = int(consumption * (config.percent / 100))
        
    return receive
    
@property
def canReceiveBonus(self):
    return calcReceiveBouns(self.total_excl_tax, self.shipping_excl_tax)
    
Order.hasReceiveBonus = hasReceiveBonus
Order.canReceiveBonus = canReceiveBonus
    
@property
def bonusRecords(self):
    return BonusRecord.objects.filter(user=self).order_by('-createdTime')
    
User.bonusRecords = bonusRecords    
    
class BonusSettings(SingletonModel):
    consumption = models.PositiveIntegerField(
        verbose_name=_("Consumption exceed(0 = no limit)"), default=0)
    percent = models.FloatField(
        verbose_name=_("Percent of consumption turn into bouns(10 = 10%)"), default=0.0,
        validators=[MinValueValidator(0.0),],
    )
    discount = models.FloatField(
        verbose_name=_("Discount per bonus"), default=0.0, 
        validators=[MinValueValidator(0.0),],
    )
    limit = models.IntegerField(
        verbose_name=_("Dividend use cap(-1 is no limit)"), default=-1,
        validators=[MinValueValidator(-1),],
    )
    
    receiveStatus = models.CharField(
        choices=settings.OSCAR_ORDER_STATUS, 
        default=settings.OSCAR_ORDER_COMPLETE_STATUS,
        max_length=50, verbose_name=_("Receive status"))
    
    includeRange = models.ForeignKey(
        'offer.Range',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
        verbose_name=_("Include range(default is all products)"),
        related_name='bonusIncludeEange'
    )
    excludeRange = models.ForeignKey(
        'offer.Range',
        blank=True,
        null=True,
        on_delete=models.CASCADE,
        verbose_name=_("Exclude range(default is no limit)"),
        related_name='bonusExcludeRange'
    )
    
# https://code.nuwainfo.com/trac/Mercurius/ticket/3301#ticket
from oscar.apps.customer.managers import CommunicationTypeManager
CommunicationEventType = get_model('customer', 'CommunicationEventType')
RawHTML = get_model('promotions', 'RawHTML')

def getATMInfo():
    bank, created = RawHTML.objects.get_or_create(
        name='ATM Bank ID',
        defaults={
            'body': '<p>XXX</p>',
        }
    )
    account, created = RawHTML.objects.get_or_create(
        name='ATM Account',
        defaults={
            'body': '<p>XXXXXXXXXXXXX</p>',
        }
    )
    return (bank, account)

def get_messages(self, ctx=None):
    """
    Return a dict of templates with the context merged in

    We look first at the field templates but fail over to
    a set of file templates that follow a conventional path.
    """
    code = self.code.lower()
    
    # Build a dict of message name to Template instances
    templates = {'subject': 'email_subject_template',
                 'body': 'email_body_template',
                 'html': 'email_body_html_template',
                 'sms': 'sms_template'}
    for name, attr_name in templates.items():
        field = getattr(self, attr_name, None)
        
        if field is not None:
            # Template content is in a model field
            templates[name] = engines['django'].from_string(field)
        else:
            # Model field is empty - look for a file template
            template_name = getattr(self, "%s_file" % attr_name) % code
            
            try:
                templates[name] = get_template(template_name)
            except TemplateDoesNotExist:
                templates[name] = None

    # Pass base URL for serving images within HTML emails
    if ctx is None:
        ctx = {}
        
    if code == 'order_placed':
        bank, account = getATMInfo()
        ctx['bank'] = bank
        ctx['account'] = account
       
    ctx['static_base_url'] = getattr(
        settings, 'OSCAR_STATIC_BASE_URL', None)

    messages = {}
    for name, template in templates.items():
        messages[name] = template.render(ctx) if template else ''

    # Ensure the email subject doesn't contain any newlines
    messages['subject'] = messages['subject'].replace("\n", "")
    messages['subject'] = messages['subject'].replace("\r", "")

    return messages

# class CommunicationTypeManager2(CommunicationTypeManager):
    # def get_and_render(self, code, context):
        # print("ABC")
        # try:
            # commtype = self.get(code=code)
        # except self.model.DoesNotExist:
            # commtype = self.model(code=code)
        # return commtype.get_messages(context)
        
CommunicationEventType.get_messages = get_messages
    
class ECPayTrade(models.Model):
    merchantId = models.CharField(max_length=255)
    merchantTradeNo = models.CharField(max_length=255)
    tradeNo = models.CharField(max_length=255)
    
    tradeDate = models.DateTimeField()
    paymentDate = models.DateTimeField()
    
    tradeAmt = models.IntegerField()
    payAmt = models.IntegerField()
         
    rtnCode = models.CharField(max_length=255)
    rtnMsg = models.CharField(max_length=255)
    paymentType = models.CharField(max_length=255)
    simulatePaid = models.CharField(max_length=255)
    paymentTypeChargeFee = models.CharField(max_length=255)

class AllPayTrade(models.Model):
    merchantId = models.CharField(max_length=255)
    merchantTradeNo = models.CharField(max_length=255)
    tradeNo = models.CharField(max_length=255)
    
    tradeDate = models.DateTimeField()
    paymentDate = models.DateTimeField()
    
    tradeAmt = models.IntegerField()
    payAmt = models.IntegerField()
         
    rtnCode = models.CharField(max_length=255)
    rtnMsg = models.CharField(max_length=255)
    paymentType = models.CharField(max_length=255)
    simulatePaid = models.CharField(max_length=255)
    paymentTypeChargeFee = models.CharField(max_length=255)

class SpgatewayTrade(models.Model):
    merchantId = models.CharField(max_length=255)
    merchantTradeNo = models.CharField(max_length=255)
    tradeNo = models.CharField(max_length=255)   
    paymentDate = models.DateTimeField()
    paymentType = models.CharField(max_length=255) 
    tradeAmt = models.IntegerField()
    rtnMsg = models.CharField(max_length=255)
    
    paymentMethod = models.CharField(max_length=255, null=True)
    rtnCode = models.CharField(max_length=255, null=True)
    
class CathaybkCreditTrade(models.Model):
    merchantId = models.CharField(max_length=255) # STOREID
    merchantTradeNo = models.CharField(max_length=255) # ORDERNUMBER
    tradeNo = models.CharField(max_length=255) # AUTHCODE
    
    paymentDate = models.DateTimeField() # AUTHTIME
    
    tradeAmt = models.IntegerField() # AMOUNT
         
    rtnCode = models.CharField(max_length=255) # AUTHSTATUS
    rtnMsg = models.CharField(max_length=255) # AUTHMSG
    cardNo = models.CharField(max_length=255) # CARDNO

class ECPayLogisticTrade(models.Model):
    # B2C & C2C
    merchantId = models.CharField(max_length=255) # MerchantID
    merchantTradeNo = models.CharField(max_length=255) # MerchantTradeNo
    tradeNo = models.CharField(max_length=255) # AllPayLogisticsID
    
    tradeDate = models.DateTimeField()
    updateStatusDate = models.DateTimeField() # UpdateStatusDate
    
    tradeAmt = models.IntegerField() # GoodsAmount
         
    rtnCode = models.CharField(max_length=255) # RtnCode
    rtnMsg = models.CharField(max_length=255) # RtnMsg

    receiverName = models.CharField(max_length=255) # ReceiverName
    receiverCellPhone = models.CharField(max_length=255) # ReceiverCellPhone
    receiverEmail = models.CharField(max_length=255, null=True) # ReceiverEmail

    logisticsType = models.CharField(max_length=255) # LogisticsType
    logisticsSubType = models.CharField(max_length=255) # LogisticsSubType

    # return
    rtnMerchantTradeNo = models.CharField(max_length=255, null=True)
    rtnOrderNo = models.CharField(max_length=255, null=True)

    # CVS C2C
    paymentNo = models.CharField(max_length=255, null=True) # CVSPaymentNo 
    validationNo = models.CharField(max_length=255, null=True) # CVSValidationNo 
    
