#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: BrowserPanel.py 9621 2016-04-06 13:36:10Z Eric $
#
# Copyright (c) 2012 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Eric $
# $Date: 2010-08-30 19:51:13 +0800$
# $Revision: 9621 $
#
# Contributors:
#  Bear (original author)
#  Yachu

"""
BrowserPanel for Trident implementation.
"""

import wx
import wx.lib.iewin as iewin

class Impl(wx.Panel):
    """
    A control to embed WebBrowser control.
    """

    def __init__(self, parent, defaultURL="about:blank",
                 navigateErrorProcessor=None,
                 documentCompleteProcessor=None,
                 progressChangeProcessor=None,
                 beforeNavigate2Processor=None,
                 navigateComplete2Processor=None,
                 titleChangeProcessor=None,
                 newWindow3Processor=None):
        """
        Constructor.

        @param parent Parent.
        @param defaultURL The URL will be loaded on initial.
        @param navigateErrorProcessor Additional event processor on navigate
                                      error.
        @param documentCompleteProcessor Additional event processor on document
                                         complete.
        """
        wx.Panel.__init__(self, parent, -1,
                          style=wx.TAB_TRAVERSAL |
                                wx.CLIP_CHILDREN |
                                wx.NO_FULL_REPAINT_ON_RESIZE)

        self.default = defaultURL
        self.current = defaultURL
        self.navigateErrorProcessor = navigateErrorProcessor
        self.documentCompleteProcessor = documentCompleteProcessor
        self.progressChangeProcessor = progressChangeProcessor
        self.beforeNavigate2Processor = beforeNavigate2Processor
        self.navigateComplete2Processor = navigateComplete2Processor
        self.titleChangeProcessor = titleChangeProcessor
        self.newWindow3Processor = newWindow3Processor

        sizer = wx.BoxSizer(wx.VERTICAL)

        self.ie = iewin.IEHtmlWindow(self, -1,
                                     style=wx.NO_FULL_REPAINT_ON_RESIZE)

        sizer.Add(self.ie, 1, wx.EXPAND)

        self.ie.LoadUrl(self.current)

        self.SetSizer(sizer)

        # Hook up the event handlers for the IE window.
        self.ie.AddEventSink(self)

    def navigate(self, url):
        """
        Navigate to specified URL.

        @param url URL.
        """
        self.ie.LoadUrl(url)

    def TitleChange(self, text):
        if self.titleChangeProcessor:
            self.titleChangeProcessor(self, text)

    def NavigateError(self, this,
                      pDisp, URL, TargetFrameName, StatusCode, Cancel):
        """
        Event handler for WebBrowser control's OnNavigateError.
        """
        Cancel[0] = True
        self.ie.LoadUrl(self.default)
        if self.navigateErrorProcessor:
            self.navigateErrorProcessor(self, URL[0], StatusCode, {
                'this': this,
                'pDisp': pDisp,
                'URL': URL,
                'TargetFrameName': TargetFrameName,
                'StatusCode': StatusCode,
                'Cancel': Cancel,
            })

    def ProgressChange(self, this, Progress, ProgressMax):
        """
        Event handler for WebBrowser control's OnProgressChange.
        """
        if self.progressChangeProcessor:
            self.progressChangeProcessor(self, {
                'this': this,
                'Progress': Progress,
                'ProgressMax': ProgressMax,
            })

        # Progress == -1 means loaded and rendered.
        if Progress == -1 and (self.ie.locationurl == self.current):
            # It must not be about:blank if user does not specified about:blank
            # as URL to navigate.
            if self.current == 'about:blank' and self.default != 'about:blank':
                return

    def NewWindow3(self, this, pDisp, Cancel, Flags, urlContext, URL):
        """
        Event handler for WebBrowser control's NewWindow3.
        """
        if self.newWindow3Processor:
            self.newWindow3Processor(self, {
                'this': this,
                'pDisp': pDisp,
                'Cancel': Cancel,
                'Flags': Flags,
                'urlContext': urlContext,
                'URL': URL,
            })

    def DocumentComplete(self, this, pDisp, URL):
        """
        Event handler for WebBrowser control's OnDocumentComplete.
        """
        self.current = URL[0]
        if self.documentCompleteProcessor and self.current != 'about:blank':
            self.documentCompleteProcessor(self, {
                'this': this,
                'pDisp': pDisp,
                'URL': URL,
            })

    def BeforeNavigate2(self, this, pDisp, URL, Flags, TargetFrameName,
                        PostData, Headers, Cancel):
        """
        Event handler for WebBrowser control's OnBeforeNavigate2.
        """
        self.current = URL[0]
        if self.beforeNavigate2Processor and self.current != 'about:blank':
            self.beforeNavigate2Processor(self, {
                'this': this,
                'pDisp': pDisp,
                'URL': URL,
                'Flags': Flags,
                'TargetFrameName': TargetFrameName,
                'PostData': PostData,
                'Headers': Headers,
                'Cancel': Cancel,
            })

    def NavigateComplete2(self, pDisp, URL):
        """
        Event handler for WebBrowser control's OnNavigateComplete2.
        """
        self.current = URL[0]
        if self.navigateComplete2Processor and self.current != 'about:blank':
            self.navigateComplete2Processor(self, {
                'pDisp': pDisp,
                'URL': URL,
            })
