#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Patch.py 12638 2020-12-17 07:19:51Z Lavender $
#
# Copyright (c) 2015 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $ (last)
# $Date: 2020-12-17 16:19:51 +0900 (週四, 17 十二月 2020) $
# $Revision: 12638 $
#
# Contributors:
#  Bear
#  Yachu

import importlib

from Iuppiter.DjangoUtil import (
    extendInstalledApps, extendMiddlewareClasses, DJANGO_VERSION, patterns)
from Iuppiter.Util import colored

from .Util import update, populateEnvToSettings

def attachDebugSettings(settingsDict):
    """
    Attach debug related settings to settings.py.
    It must be called in settings by passing locals().

    @param settingsLocals locals() in settings.py.
    """
    
    # django extensions
    import warnings
    
    try:
        import django_extensions
    except ImportError:
        warnings.warn(
            colored(
                "Can't import django_extensions."
                " Please install django-extensions.", 
                'yellow', attrs=['bold']
            )
        )
    else:
        extendInstalledApps(settingsDict, ('django_extensions',))
        
    # debug tool bar
    try:
        import debug_toolbar
    except ImportError:
        warnings.warn(
            colored(
                "Can't import debug_toolbar."
                " Please install django-debug-toolbar==1.9.1.", 
                'yellow', attrs=['bold']
            )
        )
    else:
        if settingsDict.get('DEBUG_TOOLBAR_ENABLE', None):
            extendInstalledApps(settingsDict, ('debug_toolbar',))
            extendMiddlewareClasses(
                settingsDict, 
                ("debug_toolbar.middleware.DebugToolbarMiddleware",)
            )
            
            toolbarSettings = {
                'INTERNAL_IPS': ['127.0.0.1'],
            }
            update(settingsDict, toolbarSettings)
            
            populateEnvToSettings((
                'INTERNAL_IPS',
            ), settingsDict)

def extendSettings(settingsLocals):
    """
    Extend settings.

    @param settings Django project's settings module.
    """
    pass

def extendBuiltins(settingsLocals):
    """
    Extend builtins with our own.

    @param settingsLocals locals() in settings.py.
    """
    pass

def fixDjango(settingsLocals):
    """
    Fix django related bugs.

    @param settingsLocals locals() in settings.py.
    """
    from Iuppiter.Util import colored
    
    # http://www.no-ack.org/2010/07/mysql-transactions-and-django.html
    TX_LEVEL = "SET SESSION TRANSACTION ISOLATION LEVEL READ COMMITTED"
    
    def promptTxFix():
        print('init_command for MySQL is set to %s.' % TX_LEVEL)
    
    def fixTx(options):
        if 'init_command' not in options:
            options['init_command'] = TX_LEVEL
            promptTxFix()
        else:
            if options['init_command'].strip() != TX_LEVEL:
                # http://stackoverflow.com/questions/12165534/
                # how-can-i-specify-multiple-init-commands-in-djangos-setup-file            
                import warnings
                
                refURL = ("http://www.no-ack.org/2010/07/"
                          "mysql-transactions-and-django.html")
                warnings.warn("'%s' %s\n%s" % (
                    colored(TX_LEVEL, 'yellow', attrs=['bold']),
                    colored(" shall be set in DATABASES init_command.", 
                            'white', attrs=['bold']),
                    colored("See %s for more details." % refURL, 
                            'cyan', attrs=['bold'])))    
    
    for k, v in list(settingsLocals['DATABASES'].items()):
        if v.get('ENGINE', '').endswith('mysql'):
            if 'OPTIONS' not in v:
                v['OPTIONS'] = {"init_command": TX_LEVEL}
                promptTxFix()
            else:
                fixTx(v['OPTIONS'])

    if ('DATABASE_OPTION' in settingsLocals and 
        'DATABASE_ENGINE' in settingsLocals and
        settingsLocals['DATABASE_ENGINE'].endswith('mysql')):
        fixTx(settingsLocals['DATABASE_OPTION'])

def improveDjango(settingsLocals):
    """
    Do some simple stuff to improve django's performance.

    @param settingsLocals locals() in settings.py.
    """
    # We make sure we are using Python 2.5, so partial is faster than
    # django's curry function.
    #
    # After django fix the http://code.djangoproject.com/ticket/10369
    # We got problem to replace curry while rendering templates.
    # Please see http://code.djangoproject.com/ticket/12176
    #import functools
    #from django.utils import functional
    #functional.curry = functools.partial

def extendDjango(settingsLocals):
    """
    Extend django to provide the functionality we required.

    @param settingsLocals locals() in settings.py.
    """
    pass

def extendAll(settingsLocals):
    """
    Extend all about django.

    @param settingsLocals locals() in settings.py. For backward compatible, 
                          settings module is acceptable, but not recommended.
    """
    pass
    
def attachAdminURLs(settings, urlpatterns):
    """
    Attach admin related urlpatterns.
    It must be called in urls.py by passing urlpatterns.

    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """
    pass

def attachDebugURLs(settings, urlpatterns):
    """
    Attach debug related urlpatterns.
    It must be called in urls.py by passing urlpatterns.

    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """
    from django.conf.urls import url, include

    try:
        import debug_toolbar
    except:
        pass
    else:
        if settings.DEBUG:
            urlpatterns += patterns('',
                url(r'^__debug__/', include(debug_toolbar.urls)),
            )
            
def checkImportPackage(packageName):
    """
    Check the package you want to import and 
    raise an ImportError if the package is not available.
    
    @param packageName Package name.
    @param settingsDict Settings dict.
    """
    try:
        return importlib.import_module(packageName)
    except ImportError:
        return False

def attachURLs(settings, urlpatterns):
    """
    Attach patched (for example, debug and admin...etc.) urlpatterns.
    It must be called in urls.py by passing urlpatterns.

    @param settings Django project's settings module.
    @param urlpatterns urlpatterns in urls.py.
    """
    from functools import partial

    from django.views.decorators.csrf import csrf_exempt
    from django.conf.urls import url, include
    from des import urls as des_urls

    from Iuno.cloud.logging.views import sendLog
    
    #=========================================================================
    # Iuno.cloud.
    try:
        module = checkImportPackage('Iuno.cloud')
    except ImportError:
        pass
    else:
        if module:
            import Iuno.cloud
            
    if getattr(settings, 'IUNO_CLOUD_ENABLE', None) and module:
        attachCloudKws = {}
        if hasattr(settings, 'IUNO_CLOUD_SERVICES'):
            attachCloudKws['services'] = settings.IUNO_CLOUD_SERVICES
    
        Iuno.cloud.attachURLs(settings, urlpatterns, **attachCloudKws)
    
    #==========================================================================
    addTestUrl = False
    if hasattr(settings ,'IUNO_TEST_ENABLE'):
        addTestUrl = settings.IUNO_TEST_ENABLE
        
    if addTestUrl:
        from Iuno.test.views import test
        urlpatterns += patterns('',
            url(r"^remote/test/$", test, name="remoteTest"),
        )
    
    if settings.DEBUG == False and settings.SERVER_MODE == settings.DEVELOPMENT:
        if DJANGO_VERSION >= 11100: # django >= 1.11
            from django.views.static import serve
            urlpatterns += patterns('',
                url(r'^media/(?P<path>.*)$', serve, {
                    'document_root': settings.MEDIA_ROOT,
                })
            )
        else:
            urlpatterns += patterns('',
                url(r'^media/(?P<path>.*)$', 'django.views.static.serve', {
                    'document_root': settings.MEDIA_ROOT,
                })
            )

# http://www.arthurkoziel.com/2009/01/15/
# passing-mediaurl-djangos-500-error-view/

def handle500(request, template_name='500.html'):
    """
    500 error handler.

    Templates: `500.html`
    Context:
        MEDIA_URL
            Path of static media (e.g. "media.example.org")
    """
    pass
