#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: Mailer.py 10284 2017-09-10 17:17:32Z David $
#
# Copyright (c) 2014 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: David $ (Last)
# $Date: 2013-01-30 12:13:45 +0800$
# $Revision: 10284 $

import os
import random
import mimetypes
import time
from datetime import datetime

from smtplib import SMTPRecipientsRefused

from django.core.mail import EmailMessage
from django.utils.encoding import smart_str, smart_unicode
from django.contrib.sites.models import Site
from django.template import Context, Template
from django.template.loader import render_to_string

from emencia.django.newsletter.mailer import Mailer
from emencia.django.newsletter.utils.tokens import tokenize
from emencia.django.newsletter.utils.newsletter import body_insertion
from emencia.django.newsletter.settings import TRACKING_LINKS
from emencia.django.newsletter.settings import TRACKING_IMAGE
from emencia.django.newsletter.settings import INCLUDE_UNSUBSCRIPTION

from Iuppiter.dispatch import makeRequests, NoResultsPending, NoWorkersAvailable
from Iuppiter.Logging import createLogger
from Iuppiter.Encoding import utf8

from Iuno.cloud.mail import EmailBackend
from Iuno.cloud.mail import SendMessageException, RetryAndSendException

from Theophrastus.models import ContactMailingStatus, Newsletter
from Theophrastus.newsletter_extension.models import Header, Footer
from Theophrastus.cloud.service import CloudService
from Theophrastus.Utility import getComment, trackLinks

cloudSrv = CloudService.getInstance()

class CloudSMTP(object):
    def __init__(self, newsletter, test=False, fakeTest=False):
        """
        Constructor.

        @param newsletter Newsletter instance.
        """
        self.newsletter = newsletter
        self.server = self.newsletter.server
        self.test = test
        self.fakeTest = fakeTest

    def sendMail(self, contact, subject, fromEmail, replyTo, content,
                 attachments=None):
        """
        Send mail.
        @param contact Contact instance.
        @param subject Subject.
        @param fromEmail From email.
        @param replyTo Reply to someone.
        @param content Content.
        @param attachments Attachments.
        """
        mail = None
        try:
            backend = EmailBackend(fallback=False)
            mail = EmailMessage(
                subject, content, from_email=fromEmail, to=[contact.email],
                connection=backend,
                headers={'Reply-To': replyTo})

            if attachments:
                for (title, c, maintype) in attachments:
                    mail.attach(utf8(title), c, maintype)

            mail.content_subtype = "html"
            if self.fakeTest:
                time.sleep(3)
            else:
                mail.send(fail_silently=False)

                extra = mail.extra_headers
                messageId = extra.get('message_id', '')
                requestId = extra.get('request_id', '')
                message = extra.get('message', '')

                cloudSrv.recordSendInformation(
                    self.newsletter, contact, messageId, message)

            if not self.test:
                self.newsletter.status = Newsletter.SENT
                self.newsletter.save()

            return True
        except Exception as e:
            raise e

    def quit(self):
        return

class CloudMailer(Mailer):
    def __init__(self, newsletter, pool=None, test=False, fakeMail=False,
                 fakeSMTP=False, logger=None):
        """
        Constructor.
        @param newsletter Newsletter instance.
        @param test Is test?
        """
        super(CloudMailer, self).__init__(newsletter, test=test)

        # If newsletter's SMTP server was cloud(default: id=1)
        self.useCloud = True if self.newsletter.server.id == 1 else False

        self.pool = pool

        # Create logger.
        if not logger:
            self.logger = createLogger('theophrastusMailer', 'mailer.log')
        else:
            self.logger = logger

        self.fakeMail = fakeMail
        if self.fakeMail:
            self.logger.info(
                '[%d]__init__: Open fake send mode.' % newsletter.id)

        self.fakeSMTP = fakeSMTP
        if self.fakeSMTP:
            self.logger.info(
                '[%d]__init__: Open fake SMTP mode.' % newsletter.id)

    def _getFileContent(self, path):
        """
        Get file content from path.

        @param path File's path.
        """
        with open(path, 'rb') as f:
            return f.read()

    def getAttachments(self):
        """
        Get attachments.
        """
        attachments = []
        for attachment in self.newsletter.attachment_set.all():
            filePath = attachment.file_attachment.path
            attachContent = self._getFileContent(filePath)

            if attachment.contentType:
                ctype = attachment.contentType
            else:
                ctype, encoding = mimetypes.guess_type(filePath)
                if ctype is None or encoding is not None:
                    ctype = 'application/octet-stream'

            root, ext = os.path.splitext(filePath)
            attachments.append(
                ('%s%s' % (attachment.title, ext), attachContent, ctype))

        return attachments

    @property
    def can_send(self):
        """
        Check if the newsletter can be sent
        """
        if self.newsletter.server.credits() <= 0:
            self.logger.debug(
                '[%d]_[can_send]_CANNOT: Server credits <= 0.' %
                self.newsletter.id)
            return False

        if self.test:
            return True

        if self.newsletter.sending_date <= datetime.now() and \
               (self.newsletter.status == Newsletter.WAITING):
            return True

        self.logger.info(
            ('[%d]_[can_send]_CANNOT: Newsletter(status: %d) can not send:'
             'sending_date: %s, datetime.now: %s (TZ: %s).') %
            (self.newsletter.id, self.newsletter.status,
             self.newsletter.sending_date, datetime.now(),
             os.environ.get('TZ', 'None')))

        return False

    def testSend(self, contact, wait=5):
        """
        Test send.
        We can't sent any mail only wait few seconds.

        @param contact Contact instance.
        @param wait Sleeping time, default is five seconds.
        """
        newsletterId = self.newsletter.id
        contactId = contact.id
        _s = 'Send a test mail to contact<%d>' % contactId
        try:
            if random.randint(1, 100) > 15:
                time.sleep(wait)
                # Change newsletter's status.
                self.newsletter.status = Newsletter.SENT
                self.newsletter.save()
                s = '%s success.' % _s
                self.logger.info('[%d]_TEST_SUCCESS: %s.' % (newsletterId, s))
                return True
            else:
                raise RuntimeError('Failed...')
        except Exception as e:
            s = '%s failed.' % _s
            self.logger.error('[%d]_TEST_FAILED: %s.' % (newsletterId, s))
            raise e
        finally:
            s = '%s end.' % _s
            self.logger.info('[%d]_TEST_END: %s' % (newsletterId, s))

    def _replaceContent(self, content):
        """
        Replace content.

        @param content The content of replaced.
        """
        mapping = (
            ("NEWSLETTER_SITE_LINK",
             "{{ domain }}{% url 'newsletter_newsletter_contact' "
             "slug=newsletter.slug,uidb36=uidb36 token=token %}"),
            ("http://NEWSLETTER_UNSUBSCRIBE_LINK",
             "{% if uidb36 and token %}http://{{ domain }}"
             "{% url 'newsletter_mailinglist_unsubscribe' "
             "slug=newsletter.slug uidb36=uidb36 token=token %}"
             "{% else %}#{% endif %}"),
        )

        for (key, _c) in mapping:
            content = content.replace(key, _c)
        return content

    def getHeaderContent(self):
        """
        Get content of newsletter's header.
        """
        try:
            header = Header.objects.all()[0]
            _content = self._replaceContent(header.content)
            return Template(_content)
        except:
            return None

    def getFooterContent(self):
        """
        Get content of newsletter's footer.
        """
        try:
            footer = Footer.objects.all()[0]
            _content = self._replaceContent(footer.content)
            return Template(_content)
        except:
            return None

    def buildEmailContent(self, contact):
        """
        Generate the mail content for a contact.

        @param contact Contact instance.
        """
        uidb36, token = tokenize(contact)
        context = Context({'contact': contact,
                           'domain': Site.objects.get_current().domain,
                           'newsletter': self.newsletter,
                           'uidb36': uidb36, 'token': token})

        content = self.newsletter_template.render(context)
        if TRACKING_LINKS:
            content = trackLinks(content, context)

        headerContent = self.getHeaderContent()
        if headerContent is None:
            linkSite = render_to_string(
                'newsletter/newsletter_link_site.html', context)
        else:
            linkSite = headerContent.render(context)
        content = body_insertion(content, linkSite)

        if INCLUDE_UNSUBSCRIPTION:
            footerContent = self.getFooterContent()
            if footerContent is None:
                unsubscription = render_to_string(
                    'newsletter/newsletter_link_unsubscribe.html', context)
            else:
                unsubscription = footerContent.render(context)
            content = body_insertion(content, unsubscription, end=True)

        if TRACKING_IMAGE:
            image_tracking = render_to_string(
                'newsletter/newsletter_image_tracking.html', context)
            content = body_insertion(content, image_tracking, end=True)
        return smart_unicode(content)

    def send(self, contact):
        """
        Send mail to contact.

        @param contact Contact instance.
        """
        result = False
        resultMessage = ''
        newsletterId = self.newsletter.id
        contactId = contact.id
        try:
            self.logger.info(
                '[%d]_START: Send mail to contact<%d>.' %
                (newsletterId, contactId))

            if self.fakeMail:
                result = self.testSend(contact)
                resultMessage = 'Send test mail success.'
            elif self.useCloud:
                subject = self.build_title_content(contact)
                replyTo = smart_str(self.newsletter.header_reply)
                contentHtml = self.buildEmailContent(contact)
                result = self.smtp.sendMail(
                    contact,
                    subject,
                    smart_str(self.newsletter.header_sender),
                    replyTo,
                    contentHtml,
                    attachments=self.attachments)
                resultMessage = 'Send mail success.'
            else:
                message = self.build_message(contact)
                self.smtp.sendmail(
                    smart_str(self.newsletter.header_sender),
                    contact.email,
                    message.as_string())
                result = True
                resultMessage = 'Send mail success.'

            status = self.test and ContactMailingStatus.SENT_TEST \
                     or ContactMailingStatus.SENT
            self.logger.info(
                '[%d]_SUCCESS: Send mail to contact<%d> success.' %
                (newsletterId, contactId))

        except SMTPRecipientsRefused as e:
            status = ContactMailingStatus.INVALID
            contact.valid = False
            contact.save()
            _eMsg = str(e)
            resultMessage = 'Send mail failed: %s.' % _eMsg
            self.logger.error(
                '[%d]_FAILED: Send mail to contact<%d> failed, %s.' %
                (newsletterId, contactId, _eMsg))
        except Exception as e:
            _eMsg = getComment(e)
            resultMessage = 'Send mail failed: %s.' % _eMsg
            self.logger.error(
                '[%d]_FAILED: Send mail to contact<%d> failed, %s.' %
                (newsletterId, contactId, _eMsg))

            if (isinstance(e, RetryAndSendException) or
                isinstance(e, SendMessageException)):
                try:
                    errorCode = e.exception.error_code
                    errorMessage = e.exception.error_message
                    requestId = e.exception.request_id

                    cloudSrv.recordSendInformation(
                        self.newsletter, contact, requestId, errorMessage)

                    if 'rejected' in errorCode.lower():
                        status = ContactMailingStatus.REJECTED
                    else:
                        status = ContactMailingStatus.ERROR
                except Exception as e:
                    self.logger.error(
                        '[%d]_GET_ERROR_INFO_FAILED: contact<%d>.' %
                        (newsletterId, contactId))
                    status = ContactMailingStatus.ERROR
            else:
                status = ContactMailingStatus.ERROR
        finally:
            self.logger.info(
                '[%d]_END: Send mail to contact<%d> end.' %
                (newsletterId, contactId))

        ContactMailingStatus.objects.create(
            newsletter=self.newsletter, contact=contact, status=status)

        return {'result': result , 'message': resultMessage, 'id': contactId}

    def run(self):
        """
        Override.
        Send the mails

        @param pool ThreadPool instance.
        """
        self.logger.info('doRun: %d' % self.newsletter.id)

        newsletterId = self.newsletter.id

        if not self.can_send:
            self.logger.info(
            '[%d][run]_PASS: Newsletter cannot send.' % newsletterId)
            return

        # If you want pass multiple arguments, you need give a list, this list
        # should include a tuple, like: (args, kwargs)
        # argsList = []
        # for _c in self.expedition_list:
        #     argsList.append(([_c, self.newsletter], {}))
        try:
            self.update_newsletter_status()

            if not self.smtp:
                self.smtp_connect()

            if self.useCloud:
                self.attachments = self.getAttachments()
            else:
                self.attachments = self.build_attachments()

            if self.pool:
                self.logger.info(
                    '[%d][run]_START: Preparing to put worker to pool.' %
                    newsletterId)
                # Put worker to pool.
                for req in makeRequests(self.send, self.expedition_list):
                    self.pool.putRequest(req)

                # FIXME: We should override method "poll", set timeout value for
                # get value from pool's queue.
                self.pool.poll()
                self.logger.info(
                    '[%d][run]_SUCCESS: Put worker to pool success.' %
                    newsletterId)
            else:
                self.logger.info(
                    '[%d][run]_START: Preparing to send mails.' % newsletterId)

                resposne = []
                for contact in self.expedition_list:
                    resposne.append(self.send(contact))

                self.logger.info(
                    '[%d][run]_SUCCESS: Send mails success.' % newsletterId)
                return resposne

        except Exception as e:
            if self.pool:
                _s = 'Put worker to pool failed'
            else:
                _s = 'Send mail failed'

            s = ('[%d][run]_FAILED: %s,' % (newsletterId, _s))
            if isinstance(e, NoResultsPending):
                self.logger.error('%s reason: NoResultsPending, %s.' % 
                                 (s, str(e)))
            elif isinstance(e, NoWorkersAvailable):
                self.logger.error('%s reason: NoWorkersAvailable, %s.' % 
                                 (s, str(e)))
            else:
                self.logger.error('%s reason: %s.' % (s, getComment(e)))
        finally:
                       
            if self.smtp:
                if self.pool:
                    self.pool.wait()
                self.smtp.quit()
                
            self.update_newsletter_status()                

            if self.pool:
                _s = 'Put worker to pool over.'
            else:
                _s = 'Send mail over.'
            self.logger.info(
                '[%d][run]_END: %s' % (newsletterId, _s))

    def smtp_connect(self):
        """
        Override.
        Make a connection to the SMTP
        """
        if self.useCloud:
            self.smtp = CloudSMTP(self.newsletter, test=self.test,
                                  fakeTest=self.fakeSMTP)
        else:
            self.smtp = self.newsletter.server.connect()
