#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: models.py 11641 2019-06-19 07:33:01Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2019-06-19 16:33:01 +0900 (週三, 19 六月 2019) $
# $Revision: 11641 $

import datetime

from django.db import models
from django.conf import settings
from django.core.exceptions import ValidationError
from django.utils.translation import ugettext_lazy as _
from cms.models import CMSPlugin

from Zephyrus.comment import settings as _settings

class CommentPlugin(CMSPlugin):
    useFbComment = models.BooleanField(_('Use Facebook Comment'))

    appId = models.CharField(
        _("Facebook API app ID"), max_length=20, null=True, blank=True)
    
    emailTo = models.CharField(
        _('Send form data to e-mail address'), max_length=255, blank=True,
        help_text=_('Separate several addresses with a comma.'), default=""
    )

    renderTemplate = models.CharField(
        _('Comment Template'), max_length=150,
        choices=_settings.ZEPHYRUS_COMMENT_TEMPLATES,
        default=_settings.ZEPHYRUS_COMMENT_DEFAULT_TEMPLATE,
    )

    def clean(self):
        if self.useFbComment:
            if not self.appId:
                raise ValidationError(
                        _('When use facebook comment plugin'
                        ', you should set facebook app id.')
                      )

class RecentResponsePlugin(CMSPlugin):
    limit = models.PositiveIntegerField(default=6)

    renderTemplate = models.CharField(
        _('Comment Template'), max_length=150,
        choices=_settings.ZEPHYRUS_RECENTRESPONSE_TEMPLATES,
        default=_settings.ZEPHYRUS_RECENTRESPONSE_DEFAULT_TEMPLATE,
    )

class Comment(models.Model):
    url = models.CharField(
        verbose_name=_("URL"), max_length=400, 
        editable=False, null=True, db_index=True)
    
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL, null=True, blank=True, verbose_name=_("User"),)
    email = models.EmailField(
        verbose_name=_("Email"), max_length=254, null=True, blank=True)
    
    message = models.TextField(verbose_name=_("Message"))

    createDate = models.DateTimeField(
        verbose_name=_("Create date"), auto_now_add=True)
    display = models.BooleanField(verbose_name=_("Display"), default=True)

    facebook = models.BooleanField(
        verbose_name=_("Is Facebook comment plugin"), default=False)
    facebookMsgId = models.CharField(
        verbose_name=_("Facebook comment Id"), max_length=400, 
        editable=False, null=True)
    facebookUser = models.CharField(
        verbose_name=_("Facebook user"), 
        max_length=128, editable=False, null=True)
    
    @property
    def userName(self):
        if not self.facebook:
            if self.user:
                return self.user.username
            else:
                return self.email
        else:
            return self.facebookUser

    def __unicode__(self):
        return u"<Comment %d: %s>" % (self.id, self.url)

    class Meta:
        verbose_name = _("Comment")
        verbose_name_plural = _("Comments")


class CommentConfig(models.Model):
    url = models.CharField(max_length=400, editable=False, null=True)
    useFbComment = models.BooleanField(_('Use Facebook Comment'))