#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: views.py 11616 2019-06-07 02:15:46Z Andy $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Andy $
# $Date: 2019-06-07 10:15:46 +0800 (週五, 07 六月 2019) $
# $Revision: 11616 $

import re
import time
import json

from urllib.parse import urljoin

from Iuppiter.dispatch import ThreadPool
from Iuppiter.extension.views import jsonpCallback
from Iuppiter.Logging import createLogger

from django.contrib.sites.models import Site
from django.core.urlresolvers import reverse
from django.shortcuts import HttpResponse
from django.utils.translation import ugettext as _
from django.core.cache import cache
from django.shortcuts import render_to_response, get_object_or_404
from django.template import RequestContext

from Iuppiter.extension.views import jsonpCallback
from Iuppiter.Encoding import _unicode
from Iuppiter.Logging import createDebugLogger

from django.conf import settings

from emencia.django.newsletter.models import ContactMailingStatus, Contact, \
                                             Newsletter

from Theophrastus.cloud.service import CloudService
from Theophrastus.cloud.models import ErrorMessage
from Theophrastus.cloud.Amazon import SES, SNS
from Theophrastus.cloud.Mailer import CloudMailer

cloudSrv = CloudService.getInstance()

def _getDomain():
    site = Site.objects.get_current()
    domain = site.domain.replace('.', '-')
    domain = domain.replace(':', '-')
    return domain

try:
    _domain = _getDomain()
except Exception as e:
    # Migrate...?
    print(str(e))
    _domain = ''

AWS_ACCESS_KEY_ID = settings.AWS_ACCESS_KEY_ID
AWS_SECRET_ACCESS_KEY = settings.AWS_SECRET_ACCESS_KEY


BOUNCES_TOPIC_NAME = getattr(settings, 'BOUNCES_TOPIC_NAME', 'bounces')
BOUNCES_TOPIC_NAME = '%s_%s' % (_domain, BOUNCES_TOPIC_NAME)
COMPLAINTS_TOPIC_NAME = getattr(settings, 'COMPLAINTS_TOPIC_NAME', 'complaints')
COMPLAINTS_TOPIC_NAME = '%s_%s' % (_domain, COMPLAINTS_TOPIC_NAME)
DELIVERIES_TOPIC_NAME = getattr(settings, 'DELIVERIES_TOPIC_NAME', 'deliveries')
DELIVERIES_TOPIC_NAME = '%s_%s' % (_domain, DELIVERIES_TOPIC_NAME)

SUBSCRIBE_RETRY_COUNT = getattr(settings, 'SUBSCRIBE_RETRY_COUNT', 3)

amazonLogger = createDebugLogger('amazon', settings.AMAZON_LOGFILE)

def getNameFromArn(arn):
    s = arn.split(':')
    return s[-1]

class EmailAddressException(Exception):
    def __init__(self, email):
        self.email = email

    def __str__(self):
        if self.email:
            return _("Incorrect e-mail address: %s.") % self.email
        else:
            return _("Please enter e-mail address.")

def filterEmail(email):
    """
    Filter email address.

    @param email Email address.
    """
    email = email.strip().lower()

    if '<' and '>' in email:
        EMAIL_FILTER = re.compile(r'(.*?)\<(?P<email>.*?)\>')
        filterResult = EMAIL_FILTER.search(email)
        if filterResult:
            filterContent = filterResult.groupdict()
            email = filterContent.get('email', '')
        else:
            raise EmailAddressException(email)

    if not email:
        raise EmailAddressException(email)
    return email

def setSenderByCache(sender, timeout=216000):
    """
    Set verification sender.

    @param sender Sender's email address.
    @param timeout The number of seconds the value should be stored in the
           cache.
    """
    cache.set('amazonSender_%s' % sender, True, timeout)

def checkSenderByCache(sender):
    """
    Get verification sender.

    @param sender Sender's email address.
    """
    return cache.get('amazonSender_%s' % sender, False)

def doVerifyEmailAddress(email):
    """
    Verify email address.

    Reference:
        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/index.html#
        m=AmazonSES/verify_email_address

        and

        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
        index.html#m=AmazonSES/list_verified_email_addresses

    @param email Email address.
    """
    try:
        response = {
            'verification': True,
            'continue': False,
            'message': _("This email address already verified.\n")
        }

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        if email in ses.getAllVerifiedSenders():
            checkBounce = None
            checkComplaint = None

            notifyAttrValues = ses.getIdentityNotificationAttributes(email)
            bounceTopic = notifyAttrValues.get('BounceTopic', '')
            if bounceTopic:
                checkBounce = False
                if getNameFromArn(bounceTopic) == BOUNCES_TOPIC_NAME:
                    checkBounce = True

            complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
            if complaintTopic:
                checkComplaint = False
                if getNameFromArn(complaintTopic) == COMPLAINTS_TOPIC_NAME:
                    checkComplaint = True

            deliveryTopic = notifyAttrValues.get('DeliveryTopic', '')
            if deliveryTopic:
                checkDelivery = False
                if getNameFromArn(deliveryTopic) == DELIVERIES_TOPIC_NAME:
                    checkDelivery = True

            if (checkBounce is False) or (checkComplaint is False):
                response['continue'] = True
                response['message'] += _(
                    "Your email has been set for other system.\n"
                    "Are you sure you want to continue?")
            elif (checkBounce is None) or (checkComplaint is None):
                response['message'] += _("You not yet set mail notifications.")

            return response
        else:
            try:
                ses.verifyEmailAddress(email)
                return {
                    'verification': False,
                    'message': _(
                        "You not yet verify your email address.\n"
                        "Your verify request was sent, please to receive "
                        "verify email on your sender email box.\n")
                }

            except Exception as e:
                return {
                    'verification': False,
                    'message': _("Send verify request failed, please check "
                                 "your e-mail address.\nreason: %s" % str(e))
                }
    except EmailAddressException as e:
        return {
            'verification': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'verification': False,
            'message': _("Unknown error: %s") % str(e)
        }

@jsonpCallback()
def verifyEmailAddress(request):
    """
    Verify email address.
    """
    email = filterEmail(request.GET.get('email', ''))
    return doVerifyEmailAddress(email)

@jsonpCallback()
def deleteVerifiedEmailAddress(request):
    """
    Verify email address.

    Reference: http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
               index.html#m=AmazonSES/delete_verified_email_address
    @param email Email address.
    """
    try:
        email = filterEmail(request.GET.get('email', ''))

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        response = ses.deleteVerifiedEmailAddress(email)
        return {
            'result': True,
            'message': _("This email address already delete from "
                         "verified list.")
        }
    except EmailAddressException as e:
        return {
            'result': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'result': False,
            'message': _("Unknown error: %s") % str(e)
        }

@jsonpCallback()
def checkVerifiedEmailAddress(request):
    """
    Check email address.

    Reference:
        http://docs.amazonwebservices.com/AWSSDKforPHP/latest/
        index.html#m=AmazonSES/list_verified_email_addresses
    @param email Email address.
    """
    try:
        email = filterEmail(request.GET.get('email', ''))

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        verifiedList = ses.getAllVerifiedSenders()

        if email in verifiedList:
            response = {
                'verification': True,
                'message': _("This email address already verified.\n")
            }

            checkBounce = None
            checkComplaint = None
            notifyAttrValues = ses.getIdentityNotificationAttributes(email)

            bounceTopic = notifyAttrValues.get('BounceTopic', '')
            if bounceTopic:
                if getNameFromArn(bounceTopic) == BOUNCES_TOPIC_NAME:
                    checkBounce = True

            complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
            if complaintTopic:
                if getNameFromArn(complaintTopic) == COMPLAINTS_TOPIC_NAME:
                    checkComplaint = True

            deliveryTopic = notifyAttrValues.get('DeliveryTopic', '')
            if deliveryTopic:
                if getNameFromArn(deliveryTopic) == DELIVERIES_TOPIC_NAME:
                    checkDelivery = True

            if checkBounce and checkComplaint:
                return response
            else:
                response['verification'] = False
                response['message'] += _(
                    "But you not yet set mail notifications."
                    "Please verify again before you save or add.")

            return response
        else:
            return {
                'verification': False,
                'message': _("You not yet verify your email address.\n"
                             "Please verify email before you save or add.")
            }
    except EmailAddressException as e:
        return {
            'verification': False,
            'message': _unicode(e)
        }
    except Exception as e:
        return {
            'verification': False,
            'message': _("Unknown error: %s") % str(e)
        }

from django.views.decorators.csrf import csrf_exempt

from Iuppiter.Json import serializer

def recordStatus(messageId, destination, status, messageStr):
    amazonLogger.info("START: Start record.")
    
    try:
        amazonLogger.info(
            "messageId=%s, destination=%s, status=%s" %
            (messageId, destination, status))

        obj = cloudSrv.getCloudMailInformation(
            messageId=messageId, errorOnNotFound=False)

        # Wait for cloudmailinformation set ready.
        # See https://code.nuwainfo.com/trac/mercurius/ticket/2904#comment:14
        bufferTime = 0

        while not obj and bufferTime <= 15:
            time.sleep(5)
            bufferTime += 5

            obj = cloudSrv.getCloudMailInformation(
                messageId=messageId, errorOnNotFound=False)

        if obj:
            newsletter = obj.newsletter
            contact = obj.contact
            
            for contact in Contact.objects.filter(email__in=destination):
                ContactMailingStatus.objects.create(
                    newsletter=newsletter,
                    contact=contact,
                    status=status,
                )

            cloudSrv.recordSendInformation(
                newsletter, 
                contact, 
                messageId, 
                messageStr
            )

            amazonLogger.info("SUCCESS: Record success.")
        else:
            amazonLogger.error("PASS: Not matched, record pass.")

    except Exception as e:
        amazonLogger.error("FAILED: Record failed, reason: %s" % str(e))
    finally:
        amazonLogger.info("END: Record end.")

def recordBouncedStatus(messageId, destination, messageStr):
    recordStatus(messageId, destination, ContactMailingStatus.BOUNCED, messageStr)

def recordRejectedStatus(messageId, destination, messageStr):
    recordStatus(messageId, destination, ContactMailingStatus.REJECTED, messageStr)

def recordComplaintStatus(messageId, destination, messageStr):
    recordStatus(messageId, destination, ContactMailingStatus.COMPLAINT, messageStr)

def isBounceType(bounceType, bounceSubType):
    if bounceType == 'Undetermined' and bounceSubType == 'Undetermined':
        # Bounce.
        return True
    elif bounceType == 'Permanent' and bounceSubType == 'General':
        # Bounce.
        return True
    elif bounceType == 'Permanent' and bounceSubType == 'NoEmail':
        # Bounce.
        return True
    elif bounceType == 'Transient' and bounceSubType == 'General':
        # Bounce.
        return True
    elif bounceType == 'Transient' and bounceSubType == 'MailboxFull':
        # Rejected.
        return False
    elif bounceType == 'Transient' and bounceSubType == 'MessageToolarge':
        # Rejected.
        return False
    elif bounceType == 'Transient' and bounceSubType == 'ContentRejected':
        # Rejected.
        return False
    elif bounceType == 'Transient' and bounceSubType == 'AttachmentRejected':
        # Rejected.
        return False

import sys
import traceback
from urllib.request import urlopen

def handleSNS(request):
    """
    http://docs.amazonwebservices.com/ses/latest/DeveloperGuide/
        NotificationContents.html#BounceObject
    http://docs.amazonwebservices.com/ses/latest/DeveloperGuide/
        ExampleNotifications.html
    Format:
        http://docs.amazonwebservices.com/sns/latest/gsg/json-formats.html
    """
    # Get the message type header.
    # String messagetype = request.getHeader("x-amz-sns-message-type");
    # If message doesn't have the message type header, don't process it.
    postData = request.body

    try:
        amazonLogger.info("RECEIVED: %s" % str(postData))

        #Parse the JSON message in the message body
        #and hydrate a Message object with its contents
        #so that we have easy access to the name/value pairs
        #from the JSON message.
        postData = _unicode(postData)
        postData = serializer.deserialize(postData)

        # Process the message based on type.
        topicType = postData.get('Type', '')
        if topicType == 'SubscriptionConfirmation':
            amazonLogger.info("Topic type: %s" % topicType)
            # You should make sure that this subscription is from the topic you
            # expect. Compare topicARN to your list of topics
            # that you want to enable to add this endpoint as a subscription.
            # Confirm the subscription by going to the subscribeURL location
            # and capture the return value (XML message body as a string)
            subscribeURL = postData.get('SubscribeURL', '')
            amazonLogger.info("Subscribe URL: %s" % subscribeURL)

            conn = None
            try:
                conn = urlopen(subscribeURL)
                amazonLogger.info("URL response: %s" % conn.read())
            except Exception as e:
                amazonLogger.info("Get URL response failed: %s" % str(e))
            finally:
                conn.close()

        elif topicType == 'Notification':
            amazonLogger.info("Topic type: %s" % topicType)

            message = postData.get('Message', '')

            amazonLogger.info("Message: %s" % message)
            if not isinstance(message, dict):
                message = serializer.deserialize(message)

            notificationType = message.get('notificationType', '')

            amazonLogger.info("Message type: %s" % notificationType)
            bounceDict = message.get('bounce', '')
            complaintDict = message.get('complaint', '')
            mail = message.get('mail', {})
            destination = mail.get('destination', [])
            messageId = mail.get('messageId', '')

            if re.search(r'\"(.*?)\"', str(message)):
                escapeStr = re.search(r'\"(.*?)\"', str(message)).group(1)
                messageStr = re.sub(r'\"(.*?)\"', r'$$$$$', str(message))
                messageStr = re.sub(r'\'', r'"', messageStr)
                messageStr = messageStr.replace(
                    '$$$$$', "\"" + escapeStr + "\"")
                amazonLogger.info(messageStr)
            else:
                messageStr = str(message).replace('\'', '"')
                
            messageStr = json.dumps(
                json.loads(messageStr), 
                indent=4, 
                sort_keys=True
            )
            if notificationType == 'Bounce': # Bounced.
                # Get bounce's type.
                bounceType = bounceDict.get('bounceType', '')
                bounceSubType = bounceDict.get('bounceSubType', '')
                isBounce = isBounceType(bounceType, bounceSubType)
                if isBounce:
                    # record!
                    amazonLogger.info("Got bounce, need record...")
                    recordBouncedStatus(messageId, destination, messageStr)
                else:
                    # record!
                    amazonLogger.info("Got rejected, need record...")
                    recordRejectedStatus(messageId, destination, messageStr)

            elif notificationType == 'Complaint': # Complaints.
                # Record.
                amazonLogger.info("Got complaint, need record...")
                recordComplaintStatus(messageId, destination, messageStr)

            elif notificationType == 'Delivery': # Deliveries.
                amazonLogger.info("Got delivery, check if need record...")
                delivery = message.get('delivery', '')
                smtpResponse = delivery.get('smtpResponse', '')

                errorCodeList = ['450', '550', '553']
                if any(x in smtpResponse for x in errorCodeList):
                    amazonLogger.info("Got delivery error, need record...")
                    recordRejectedStatus(messageId, destination, messageStr)


        elif topicType == 'UnsubscribeConfirmation':
            amazonLogger.info("Topic type: %s" % topicType)
            # Handle UnsubscribeConfirmation message.
            # For example, take action if unsubscribing should not have occurred.
            # You can read the SubscribeURL from this message and
            # re-subscribe the endpoint.
            pass
        else:
            # Handle unknown message type.
            amazonLogger.info("Topic type: unknown.")
            pass
    except Exception as e:
        c = '\n'.join(traceback.format_tb(sys.exc_info()[-1]))
        amazonLogger.info("RECEIVED_FAILED: %s\n%s\n" % (c, e))

    return HttpResponse('ok')

@csrf_exempt
def getBounces(request):
    return handleSNS(request)

@csrf_exempt
def getComplains(request):
    return handleSNS(request)

@csrf_exempt
def getDeliveries(request):
    return handleSNS(request)

def checkNotificationTopic(email, useCache=False):
    """
    Check notification topic.

    @param email Email address.
    @param useCache Save verification result to cache?
    """
    try:
        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)

        notifyAttrValues = ses.getIdentityNotificationAttributes(email)

        bounceTopic = notifyAttrValues.get('BounceTopic', '')
        if getNameFromArn(bounceTopic) != BOUNCES_TOPIC_NAME:
            bounceTopic = ''

        complaintTopic = notifyAttrValues.get('ComplaintTopic', '')
        if getNameFromArn(complaintTopic) != COMPLAINTS_TOPIC_NAME:
            complaintTopic = ''

        deliveryTopic = notifyAttrValues.get('DeliveryTopic', '')
        if getNameFromArn(deliveryTopic) != DELIVERIES_TOPIC_NAME:
            deliveryTopic = ''

        if bounceTopic and complaintTopic and deliveryTopic:
            if useCache:
                # Set cache.
                setSenderByCache(email)
            return {'result': True}
        else:
            return {'result': False}
    except Exception as e:
        return {
            'result': False,
            'message': _('Check notification topic failed, reason: %s. '
                         'If this error shown again, please contact us.') %
                         _unicode(e)
        }

def doSetNotificationTopic(email, useCache=False):
    """
    Sets the Amazon SNS topic to which Amazon SES will publish bounce and
    complaint notifications for emails sent with that identity as the Source.
    Publishing to topics may only be disabled when feedback forwarding is
    enabled.
    (Given an identity (email address or domain).)

    Reference: http://docs.amazonwebservices.com/ses/latest/APIReference/
               http://docs.aws.amazon.com/sns/latest/api/
               http://sesblog.amazon.com/post/TxJE1JNZ6T9JXK/Handling-Bounces-
                      and-Complaints

    @param email Email address.
    @param useCache Save verification result to cache?
    """
    try:
        bounceTopic = ''
        complaintTopic = ''
        deliveryTopic = ''

        ses = SES(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)
        # Check topics.
        sns = SNS(AWS_ACCESS_KEY_ID, AWS_SECRET_ACCESS_KEY)

        for topic in sns.getAllTopicArns():
            if bounceTopic and complaintTopic and deliveryTopic:
                break
            topicName = getNameFromArn(topic)
            if topicName == BOUNCES_TOPIC_NAME:
                bounceTopic = topic
            if topicName == COMPLAINTS_TOPIC_NAME:
                complaintTopic = topic
            if topicName == DELIVERIES_TOPIC_NAME:
                deliveryTopic = topic

        if not bounceTopic: # Create a topic of bounces.
            bounceTopic = sns.createTopic(BOUNCES_TOPIC_NAME)

        if not complaintTopic: # Create a topic of complaints.
            complaintTopic = sns.createTopic(COMPLAINTS_TOPIC_NAME)

        if not deliveryTopic: # Create a topic of complaints.
            deliveryTopic = sns.createTopic(DELIVERIES_TOPIC_NAME)

        # Check subscription.
        hasBounceTopicExist = False
        hasComplaintTopicExist = False
        hasDeliveryTopicExist = False
        for n in range(SUBSCRIBE_RETRY_COUNT):
            if not hasBounceTopicExist:
                hasBounceTopicExist = sns.checkSubscriptionExist(
                    bounceTopic, BOUNCES_TOPIC_NAME)
            if not hasComplaintTopicExist:
                hasComplaintTopicExist = sns.checkSubscriptionExist(
                    complaintTopic, COMPLAINTS_TOPIC_NAME)
            if not hasDeliveryTopicExist:
                hasDeliveryTopicExist = sns.checkSubscriptionExist(
                    deliveryTopic, DELIVERIES_TOPIC_NAME)

        # Subscribe to a Topic.
        if settings.SECURE_PROXY_SSL_HEADER:
            protocol = settings.SECURE_PROXY_SSL_HEADER[1]
        else:
            protocol = 'http'

        bounceEndpoint = reverse(getBounces)
        if not hasBounceTopicExist:
            sns.subscribe(
                bounceTopic, protocol,
                urljoin(settings.LOCAL_URL, bounceEndpoint))

        complaintEndpoint = reverse(getComplains)
        if not hasComplaintTopicExist:
            sns.subscribe(
                complaintTopic, protocol,
                urljoin(settings.LOCAL_URL, complaintEndpoint))

        deliveryEndpoint = reverse(getDeliveries)
        if not hasDeliveryTopicExist:
            sns.subscribe(
                deliveryTopic, protocol,
                urljoin(settings.LOCAL_URL, deliveryEndpoint))

        # Set notification topic.
        ses.setIdentityNotificationTopic('Bounce', email, bounceTopic)
        ses.setIdentityNotificationTopic('Complaint', email, complaintTopic)
        ses.setIdentityNotificationTopic('Delivery', email, deliveryTopic)
        if useCache:
            # Set cache.
            setSenderByCache(email)
        return {
            'result': True,
            'bounceTopic': bounceTopic,
            'complaintTopic': complaintTopic,
            'deliveryTopic': deliveryTopic
        }
    except Exception as e:
        return {
            'result': False,
            'message': _('Set notification topic failed, reason: %s. '
                         'If this error shown again, please contact us.') %
                         _unicode(e)
        }

@jsonpCallback()
def setNotificationTopic(request):
    """
    Sets the Amazon SNS topic to which Amazon SES will publish bounce and
    complaint notifications for emails sent with that identity as the Source.
    Publishing to topics may only be disabled when feedback forwarding is
    enabled.
    """
    email = filterEmail(request.GET.get('email', ''))
    _checkResult = checkNotificationTopic(email)
    if _checkResult['result']:
        return _checkResult
    else:
        return doSetNotificationTopic(email)

def resendErrorContact(request, newsletterId):
    """
    Resend newsletter to the contacts which has error when
    sending mail previously.

    @param request Request instance.
    """
    SENDBOX_THREAD_POOL_COUNT = settings.SENDBOX_THREAD_POOL_COUNT
    ENABLE_FAKE_MAIL = settings.ENABLE_FAKE_MAIL
    ENABLE_FAKE_SMTP = settings.ENABLE_FAKE_SMTP
    LOG_LEVEL = settings.LOG_LEVEL
    MAILER_LOGFILE = settings.MAILER_LOGFILE

    sendboxPool = ThreadPool(SENDBOX_THREAD_POOL_COUNT)

    mailerLogger = createLogger(
                'theophrastusMailer', MAILER_LOGFILE, loggingLevel=LOG_LEVEL)

    newsletter = get_object_or_404(Newsletter, id=newsletterId)
    opts = Newsletter._meta

    context = {'title': _('Historic of %s') % newsletter.__unicode__(),
               'original': newsletter,
               'opts': opts,
               'object_id': newsletter.pk,
               'app_label': opts.app_label}

    # TODO: set --> distinct()
    sentContactIdList = set(
        ContactMailingStatus.objects.filter(
            newsletter_id=newsletter.id, status=ContactMailingStatus.SENT
        ).values_list('contact_id').distinct())

    contactIdSet = set(
        ContactMailingStatus.objects.filter(
            newsletter_id=newsletter.id, 
            status=ContactMailingStatus.ERROR
        ).exclude(
                contact_id__in=sentContactIdList).values_list('contact_id'))

    if contactIdSet:
        contactIdList = [ele[0] for ele in list(contactIdSet)]
        contactList = Contact.objects.filter(id__in=contactIdList)

        mailer = CloudMailer(newsletter, pool=sendboxPool,
                                        fakeMail=ENABLE_FAKE_MAIL,
                                        fakeSMTP=ENABLE_FAKE_SMTP,
                                        logger=mailerLogger)

        mailer.useCloud = True if newsletter.server.id == 1 else False
        mailer.smtp_connect()
        mailer.attachments = mailer.getAttachments()

        for contact in contactList:
            mailer.send(contact)

        context['sendConfirm'] = _('Resend error emails finish')
    else:
        context['information'] = _('All errors have been resent')

    return render_to_response('NewsletterHistoric.html',
                              context, context_instance=RequestContext(request))

def viewErrorDetails(request, slug, contactMailingStatusId):
    """
    Display the error message of a newsletter contact.

    @param request Request instance.
    @param slug Newsletter's slug.
    @param cloudmailinformationId Id of CloudMailInformation.
    """

    contactMailingStatus = ContactMailingStatus.objects.filter(
                                id=contactMailingStatusId)[0]
    opts = Newsletter._meta
    cloudSrv = CloudService.getInstance()
    newsletter = get_object_or_404(Newsletter, slug=slug)

    if contactMailingStatus.status == ContactMailingStatus.ERROR:
        content = ErrorMessage.objects.filter(
            newsletter=contactMailingStatus.newsletter,
            contact=contactMailingStatus.contact
        )[0].message
    else:
        content = cloudSrv.getCloudMailInformation(
            newsletter_id=contactMailingStatus.newsletter.id,
            contact_id=contactMailingStatus.contact.id,
            errorOnNotFound=False
        ).message
    
    context = {
        'title': _('Error of contact: %s') % contactMailingStatus.contact.email,
        'original': newsletter,
        'opts': opts,
        'object_id': newsletter.pk,
        'app_label': opts.app_label,
        'content': content
    }
    return render_to_response(
        'NewsletterHistoricErrorDetail.html',
        context,
        context_instance=RequestContext(request)
    )