#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: models.py 12420 2020-07-23 08:06:48Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-07-23 16:06:48 +0800 (週四, 23 七月 2020) $
# $Revision: 12420 $

from functools import wraps
from django import forms
from django.db import models
from django.conf import settings
from django.template import RequestContext
from django.utils.translation import ugettext_lazy as _
from cms.models import CMSPlugin
from djangocms_forms.models import FormDefinition
from filer.fields.file import FilerFileField
from aldryn_categories.models import Category
from aldryn_newsblog.models import Article
from aldryn_newsblog import models as newsblogModels
from taggit.models import Tag

from Iuppiter.DjangoUtil import DJANGO_VERSION
from Zephyrus.blog_extension import settings as _settings

METHODS = [
    ('or',  _('OR')),
    ('and',  _('AND')),
]

class FilterPlugin(CMSPlugin):
    limit = models.PositiveIntegerField(_('Limit'), default=6)

    filterMethod = models.CharField(
        _('Filter Method'), max_length=3, choices=METHODS, default='or')

    tag = models.ManyToManyField(
        Tag,
        verbose_name=_('Tag'),
        default=None,
        blank=True,
        help_text=_(
            "If you check 'Not use tags to filter article'. "
            "It will no effect.")
    )

    catagory = models.ForeignKey(
        Category,
        verbose_name=_('Catagory'),
        on_delete=models.CASCADE,
        related_name='+',
        default=None,
        blank=True,
        null=True,
        help_text=_(
            "If you check 'Not use catagories to filter articles'. "
            "It will no effect.")
    )

    allTag = models.BooleanField(
        _('Not use tags to filter articles'), 
        default=True,
        help_text=_(
            'If you check this. It will not filter articles with tags.')
    )
    allCatagory = models.BooleanField(
        _('Not use catagories to filter articles'), 
        default=True,
        help_text=_(
            'If you check this. It will not filter articles with catagories.')
    )

    renderTemplate = models.CharField(
        _('Render template'), max_length=150,
        choices=_settings.ZEPHYRUS_BLOG_MAYLIKE_TEMPLATES,
        default=_settings.ZEPHYRUS_BLOG_MAYLIKE_DEFAULT_TEMPLATE,
    )

class DatePluginConfig(CMSPlugin):
    renderTemplate = models.CharField(
        _('Render template'), max_length=150,
        choices=_settings.ZEPHYRUS_BLOG_DATE_TEMPLATES,
        default=_settings.ZEPHYRUS_BLOG_DATE_DEFAULT_TEMPLATE,
    )

# modify slug field formfield

# django == 1.11
from django import VERSION

if DJANGO_VERSION >= 11100: # 1.11
    def formfieldDecorator(f):
        @wraps(f)
        def wrapper(*args, **kwds):
            return f(
                form_class=forms.SlugField , allow_unicode=True, *args, **kwds)
        return wrapper
        
    articleFields = Article.translations.rel.related_model._meta.fields
    categoryFields = Category.translations.rel.related_model._meta.fields
else:
    def formfieldDecorator(f):
        @wraps(f)
        def wrapper(*args, **kwds):
            return f(form_class=forms.CharField, *args, **kwds)
        return wrapper
    articleFields = Article.translations.related.related_model._meta.fields
    categoryFields = Category.translations.related.related_model._meta.fields

for field in articleFields:
    if field.attname == 'slug':
        field.default_validators = []
        field.validators = []
        field.formfield = formfieldDecorator(field.formfield)

for field in categoryFields:
    if field.attname == 'slug':
        field.default_validators = []
        field.validators = []
        field.formfield = formfieldDecorator(field.formfield)

for field in Tag._meta.fields:
    if field.attname == 'slug':
        field.default_validators = []
        field.validators = []
        field.formfield = formfieldDecorator(field.formfield)

# https://code.nuwainfo.com/trac/Mercurius/ticket/3134#comment:1

from aldryn_newsblog.utils import utilities
from aldryn_newsblog.utils.utilities import get_cleaned_bits, get_field_value

def get_plugin_index_data(base_plugin, request):
    # copy from  aldryn_newsblog.utils.utilities.get_plugin_index_data

    text_bits = []

    instance, plugin_type = base_plugin.get_plugin_instance()

    if instance is None:
        # this is an empty plugin
        return text_bits

    search_fields = getattr(instance, 'search_fields', [])

    if hasattr(instance, 'search_fulltext'):
        # check if the plugin instance has search enabled
        search_contents = instance.search_fulltext
    elif hasattr(base_plugin, 'search_fulltext'):
        # now check in the base plugin instance (CMSPlugin)
        search_contents = base_plugin.search_fulltext
    elif hasattr(plugin_type, 'search_fulltext'):
        # last check in the plugin class (CMSPluginBase)
        search_contents = plugin_type.search_fulltext
    else:
        # disabled if there's search fields defined,
        # otherwise it's enabled.
        search_contents = not bool(search_fields)
    
    if search_contents:
        if hasattr(instance, 'render_plugin'):
            plugin_contents = instance.render_plugin(
                context=RequestContext(request))

            if plugin_contents:
                text_bits = get_cleaned_bits(plugin_contents)
    else:
        values = (get_field_value(instance, field) for field in search_fields)

        for value in values:
            cleaned_bits = get_cleaned_bits(value or '')
            text_bits.extend(cleaned_bits)
    return text_bits

utilities.get_plugin_index_data = get_plugin_index_data
newsblogModels.get_plugin_index_data = get_plugin_index_data


# CMSPlugin FormDefinition
class CallToActionModel(CMSPlugin):
    
    targetFile = FilerFileField(
        verbose_name=_('Shared file'),
        null=True,
        blank=True,
        on_delete=models.SET_NULL,
    )
    linkUrl = models.URLField(
        verbose_name=_('External URL'),
        blank=True,
        max_length=2040,
        help_text=_('The file link to an external URL.'),
    )
    renderTemplate = models.CharField(
        _('Call To Action Template'), max_length=150, blank=True,
        choices=_settings.CALL_TO_ACTION_TEMPLATES,
        default=_settings.CALL_TO_ACTION_DEFAULT_TEMPLATE,
    )
    fileName = models.CharField(
        _('File Name'), max_length=150, blank=True,
    )
    
    @property
    def isArticlePDF(self):
        if self.linkUrl or self.targetFile:
            return False
        else:
            return True
            


class CallToActionStatistics(models.Model):
    action = models.ForeignKey(CallToActionModel, on_delete=models.CASCADE)
    token = models.CharField(max_length=80)
    email = models.CharField(max_length=500)
    count = models.IntegerField(default=0)
    url = models.CharField(max_length=500, blank=True, null=True)
    article = models.ForeignKey(Article, on_delete=models.CASCADE, related_name='callToActionStatistics', null=True)
    
   

