#!/usr/bin/env python
# -*- coding: utf-8 -*-
# $Id: BuildBoilerplateTest.py 12546 2020-10-08 07:41:14Z Lavender $
#
# Copyright (c) 2017 Nuwa Information Co., Ltd, All Rights Reserved.
#
# Licensed under the Proprietary License,
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at our web site.
#
# See the License for the specific language governing permissions and
# limitations under the License.
#
# $Author: Lavender $
# $Date: 2020-10-08 15:41:14 +0800 (週四, 08 十月 2020) $
# $Revision: 12546 $

import os
import sys
import unittest
import logging
import shutil

from Zephyrus.boilerplate.management.commands import open3 as open

logger = logging.getLogger(__name__)
logger.setLevel(logging.DEBUG)
handler = logging.StreamHandler()
handler.setFormatter(
    logging.Formatter("%(asctime)s - %(levelname)s - %(message)s"))
logger.addHandler(handler)

HTML_DIR_PATH = os.path.abspath(
    os.path.join(__file__, '..', 'HtmlTemp'))

TEST_DIR_PATH = os.path.abspath(
    os.path.join(__file__, '..', 'TestHtml'))

TEST_STATIC_DIR_PATH = os.path.abspath(
    os.path.join(__file__, '..', 'static'))

STATIC_DIT_PATH = os.path.abspath(
    os.path.join(__file__, '..', '..', 'static'))

TEMPLATES_DIR_PATH = os.path.abspath(
    os.path.join(__file__, '..', '..', 'templates'))

ERROR_HTML_DIR = os.path.abspath(
    os.path.join(__file__, '..', 'BackupHtml'))

class BuildBoilerplateTest(unittest.TestCase):

    def runBuildBoilerplate(self, origin, answer, ver=2,
                            dst=TEST_DIR_PATH, delete=True, rename=None):
        fileFrom = os.path.join(HTML_DIR_PATH, origin)
        if rename:
            copyTo = os.path.join(dst, rename)
        else:
            copyTo = os.path.join(dst, origin)
        answer = os.path.join(HTML_DIR_PATH, answer)
        shutil.copyfile(fileFrom, copyTo)
    
        if ver == 1:
            logger.warn('Ver 1 is not existed.')
            ver = 2
        if ver == 2:
            os.system(
                "%s Manage.py build_boilerplate2 %s" % (sys.executable, 
                                                        dst))
        else:
            return

        with open(answer, 'r') as f:
            html1 = f.read().replace('\r\n', '\n')
        with open(copyTo, 'r') as f:
            html2 = f.read().replace('\r\n', '\n')
        
        try:
            self.assertEqual(html1, html2)
        except Exception as e:
            shutil.copyfile(copyTo, os.path.join(ERROR_HTML_DIR, origin))
            raise e

        if delete:
            os.remove(copyTo)

    def reset(self):
        os.system("del Zephyrus.db")
        os.system("copy _Zephyrus.db Zephyrus.db")

        try:
            shutil.rmtree(TEMPLATES_DIR_PATH)
        except Exception as e:
            os.mkdir(TEMPLATES_DIR_PATH)
        else:
            os.mkdir(TEMPLATES_DIR_PATH)

        try:
            shutil.rmtree(STATIC_DIT_PATH)
        except Exception as e:
            logger.error("Static folder is not exist.")
        else:
            os.mkdir(STATIC_DIT_PATH)

    def getTestStatic(self, path):
        self.clearStaticDir()
        path = os.path.join(TEST_STATIC_DIR_PATH, path)
        for dirName in os.listdir(path):
            dirPath = os.path.join(path, dirName)
            dst = os.path.join(STATIC_DIT_PATH, dirName)
            if os.path.isdir(dirPath):
                shutil.copytree(dirPath, dst)
            else:
                shutil.copyfile(dirPath, dst)
                
    def getTestTemplate(self, path):
        path = os.path.join(HTML_DIR_PATH, path)
        for dirName in os.listdir(path):
            dirPath = os.path.join(path, dirName)
            dst = os.path.join(TEMPLATES_DIR_PATH, dirName)
            if os.path.isdir(dirPath):
                shutil.copytree(dirPath, dst)
            else:
                shutil.copyfile(dirPath, dst)
                
    def compareTemplate(self, temp1, temp2):
        with open(temp1, 'r') as f:
            html1 = f.read().replace('\r\n', '\n')
        with open(temp2, 'r') as f:
            html2 = f.read().replace('\r\n', '\n')
            
        self.assertEqual(html1, html2)
        

    def clearStaticDir(self):
        try:
            shutil.rmtree(STATIC_DIT_PATH)
        except Exception as e:
            logger.error("Static folder is not exist.")
        else:
            os.mkdir(STATIC_DIT_PATH)

    
    def testComment(self):
        self.runBuildBoilerplate('Comment.html', 'CommentAnswer.html', ver=2)

    
    def testDiv(self):
        self.runBuildBoilerplate('Div.html', 'DivAnswer.html', ver=2)
    
    def testSpecialWord(self):
        self.runBuildBoilerplate(
            'SpecialWord.html', 'SpecialWordAnswer.html', ver=2)

    def testScriptTag(self):
        self.runBuildBoilerplate(
            'ScriptTag.html', 'ScriptTagAnswer.html', ver=2)
    
    def testStatic(self):
        self.getTestStatic('TestStatic')
        self.runBuildBoilerplate('Static.html', 'StaticAnswer.html', ver=2)

    def testTags(self):
        self.runBuildBoilerplate('Tags.html', 'TagsAnswer.html', ver=2)

    def testNav(self):
        self.runBuildBoilerplate('Nav.html', 'NavAnswer.html', ver=2)

    def testDb(self):
        self.reset()
        self.getTestStatic('Demo')
        self.runBuildBoilerplate('Demo.html', 'DemoAnswer.html', 
                                 dst=TEMPLATES_DIR_PATH, delete=False, 
                                 rename='index.html', ver=2)
        os.system("%s Manage.py create_pages" % sys.executable)
        os.system("%s Manage.py pluginize" % sys.executable)
        
        import sqlite3
        conn = sqlite3.connect('Zephyrus.db')
        c = conn.cursor()
        c.execute("SELECT COUNT(id) FROM cms_placeholder")
        placeholderCount = c.fetchall()[0][0]

        c.execute("SELECT COUNT(id) FROM cms_cmsplugin")
        pluginCount = c.fetchall()[0][0]
        c.close()
        conn.close()
        self.assertEqual(placeholderCount, pluginCount)
    
    def testNested(self):
        self.reset() 
        self.getTestStatic('Template1')
        self.getTestTemplate('Template1')
        
        os.system(
            "%s Manage.py build_boilerplate2 %s" % (sys.executable, 
                                                    TEMPLATES_DIR_PATH))
                                                    
        # index.html
        templatePath = os.path.join(
            TEMPLATES_DIR_PATH, 'index.html')
        answerPath = os.path.join(
            HTML_DIR_PATH, 'Template1Answer', 'index.html')
            
        self.compareTemplate(templatePath, answerPath)
        
        # blog/blog.html
        templatePath = os.path.join(
            TEMPLATES_DIR_PATH, 'blog', 'blog.html')
        answerPath = os.path.join(
            HTML_DIR_PATH, 'Template1Answer', 'blog', 'blog.html')
            
        self.compareTemplate(templatePath, answerPath)
        
        
        
    
    # Z只是要讓這個函數最後跑
    def testZAttr(self):
        self.reset()
        self.runBuildBoilerplate('Attr.html', 'AttrAnswer.html', 
                                 dst=TEMPLATES_DIR_PATH, delete=False, 
                                 rename='index.html', ver=2)
        os.system("%s Manage.py create_pages" % sys.executable)
        os.system("%s Manage.py pluginize" % sys.executable)

        with open(
            os.path.join(STATIC_DIT_PATH, 'js', 'FixTemplate.js'), 'r') as f:
            js1 = f.read().replace('\r\n', '\n')
        with open(os.path.join(HTML_DIR_PATH, 'AttrJsAnswer.js'), 'r') as f:
            js2 = f.read().replace('\r\n', '\n')

        self.assertEqual(js1, js2)

        import sqlite3
        conn = sqlite3.connect('Zephyrus.db')
        c = conn.cursor()
        c.execute(
            "SELECT count(link_url) FROM djangocms_picture_picture "
            "where link_url = \"http://127.0.0.1:1234\";")
        pluginCount = c.fetchall()[0][0]
        c.close()
        conn.close()

        # 因為 jenkins 跑兩次的關係導致 db 資料變兩倍
        # 因此判斷有沒有 JOB_NAME (代表在 jenkins 上)
        if 'JOB_NAME' in os.environ:
            self.assertEqual(pluginCount, 3)
        else:
            self.assertEqual(pluginCount, 2)
        self.reset()
     
    
if __name__ == '__main__':
    unittest.main()
    